\documentstyle[comments]{cweb}
@s integer_matrix int
@s integer_vector int
@i /KM/usr/mehlhorn/Programs/typnamen
\begin{document}


\title{Integer-valued Matrices}
\author{}
\maketitle


@ We discuss the implementation of the class |integer_matrix| in 
two parts. In the first part we give the
implementation of the constructors, the destructor, assignment, equality, all
access functions, and all arithmetic operators, and in the second part we
deal with the functions of linear algebra. The first part is taken from the
implementation of real-valued matrices.




A matrix is 
represented
as a C++-array of pointers to row-vectors with integer 
entries (type |integer_vector|). Thus |v[i]| is a 
pointer to a vector
and hence |v[i]->v[j]| is the entry $(i,j)$ of the matrix. 

@c
#include "integer_matrix.h"
#include <math.h>
#include <LEDA/array.h>
@<basic functions@>;
@<linear algebra@>;

@ A matrix is 
represented
as a C++-array of pointers to row-vectors with integer 
entries (type |integer_vector|). Thus |v[i]| is a 
pointer to a vector
and hence |v[i]->[j]| is the entry $(i,j)$ of the matrix. 

@<basic functions@>=





void integer_matrix::flip_rows(int i,int j)  //relict
{ integer_vector* p = v[i];
  v[i] = v[j];
  v[j] = p;
 }


integer_matrix::integer_matrix(int dim1, int dim2)  
{
  if (dim1<0 || dim2<0) 
  error_handler(1,"matrix: negative dimension."); 

  d1=dim1; 
  d2=dim2; 

  if (d1 > 0) 
  { v = new integer_vector*[d1];
    for (int i=0;i<d1;i++) v[i] = new integer_vector(d2); 
   }
  else v = nil;
}


integer_matrix::integer_matrix(const integer_matrix& p)  
{ 
  d1 = p.d1;
  d2 = p.d2;
    
  if (d1 > 0) 
  { v = new integer_vector*[d1];
    for (int i=0;i<d1;i++) v[i] = new integer_vector(*p.v[i]); 
   }
  else v = nil;
}

integer_matrix::integer_matrix(const array<integer_vector> & A)  
{   int low = A.low(); int high = A.high();
   d2 = high - low + 1; 
   d1 = A[low].dim();

    
  if (d1 > 0) 
  { v = new integer_vector*[d1];
    for (int i=0;i<d1;i++) 
	{v[i] = new integer_vector(d2); 
         for (int j = 0; j < d2; j++) v[i]->v[j] = A[low + j][i];
        }
   }
  else v = nil;
}

integer_matrix::integer_matrix(int dim1, int dim2, number** p)  
{ d1=dim1; 
  d2=dim2; 
  v = new integer_vector*[d1];
  for(int i=0;i<d1;i++) 
  { v[i] = new integer_vector(d2); 
    for(int j=0;j<d2;j++) elem(i,j) = p[i][j];
   }

 }

integer_matrix::~integer_matrix()  
{ if (v) 
  { while(d1--) delete v[d1]; 
    delete v;
   }
}


void integer_matrix::check_dimensions(const integer_matrix& mat) const
{ if (d1 != mat.d1 || d2 != mat.d2)
   error_handler(1,"incompatible matrix types.");
 }

integer_matrix::integer_matrix(const integer_vector& vec)
{ d1 = vec.d;
  d2 = 1;
  v = new integer_vector*[d1];
  for(int i=0; i<d1; i++)
  { v[i] = new integer_vector(1);
    elem(i,0) = vec[i];
   }
    
}



integer_matrix& integer_matrix::operator=(const integer_matrix& mat)
{ register int i,j;

  if (d1 != mat.d1 || d2 != mat.d2)
  { for(i=0;i<d1;i++) delete v[i];
    delete v;
    d1 = mat.d1;
    d2 = mat.d2;
    v = new integer_vector*[d1];
    for(i=0;i<d1;i++) v[i] = new integer_vector(d2);
   }

  for(i=0;i<d1;i++)
    for(j=0;j<d2;j++) elem(i,j) = mat.elem(i,j);

  return *this;
}

int integer_matrix::operator==(const integer_matrix& x) const
{ register int i,j;
  if (d1 != x.d1 || d2 != x.d2) return false;

  for(i=0;i<d1;i++)
    for(j=0;j<d2;j++)
      if (elem(i,j) != x.elem(i,j)) return false;

  return true;
 }

integer_vector integer_matrix::col(int i)  const
{ if ( i<0 || i>=d2 )  error_handler(1,"matrix: col index out of range");
  integer_vector result(d1);
  int j = d1;
  while (j--) result.v[j] = elem(j,i);
  return result;
}

integer_matrix::operator integer_vector() const
{ if (d2!=1) 
   error_handler(1,"error: cannot make integer_vector from matrix\n");
  return col(0);
}

integer_matrix integer_matrix::operator+(const integer_matrix& mat)
{ register int i,j;
  check_dimensions(mat);
  integer_matrix result(d1,d2);
  for(i=0;i<d1;i++)
   for(j=0;j<d2;j++)
      result.elem(i,j) = elem(i,j) + mat.elem(i,j);
  return result;
}

integer_matrix& integer_matrix::operator+=(const integer_matrix& mat) 
{ register int i,j;
  check_dimensions(mat);
  for(i=0;i<d1;i++)
   for(j=0;j<d2;j++)
      elem(i,j) += mat.elem(i,j);
  return *this;
}

integer_matrix& integer_matrix::operator-=(const integer_matrix& mat) 
{ register int i,j;
  check_dimensions(mat);
  for(i=0;i<d1;i++)
   for(j=0;j<d2;j++)
      elem(i,j) -= mat.elem(i,j);
  return *this;
}


integer_matrix integer_matrix::operator-(const integer_matrix& mat)
{ register int i,j;
  check_dimensions(mat);
  integer_matrix result(d1,d2);
  for(i=0;i<d1;i++)
   for(j=0;j<d2;j++)
      result.elem(i,j) = elem(i,j) - mat.elem(i,j);
  return result;
}


integer_matrix integer_matrix::operator-()  // unary
{ register int i,j;
  integer_matrix result(d1,d2);
  for(i=0;i<d1;i++)
   for(j=0;j<d2;j++)
      result.elem(i,j) = -elem(i,j);
  return result;
}


integer_matrix integer_matrix::operator*(number f) const
{ register int i,j;
  integer_matrix result(d1,d2);
  for(i=0;i<d1;i++)
   for(j=0;j<d2;j++)
      result.elem(i,j) = elem(i,j) *f;
  return result;
}

integer_matrix integer_matrix::operator*(const integer_matrix& mat) const
{ if (d2!=mat.d1)
     error_handler(1,"matrix multiplication: incompatible matrix types\n");
  
  integer_matrix result(d1, mat.d2);
  register int i,j;

  for (i=0;i<mat.d2;i++)
  for (j=0;j<d1;j++) result.elem(j,i) = *v[j] * mat.col(i);

 return result;

}


    

ostream& operator<<(ostream& s, const integer_matrix& M)
{ int i;
  s <<"\n";
  for (i=0;i<M.d1;i++) s << M[i] << "\n"; 
  return s;
}

istream& operator>>(istream& s, integer_matrix& M)
{ int i=0;
  while (i<M.d1 && s >> M[i++]);
  return s;
}


integer_matrix identity(int n)
{integer_matrix result(n,n);

 for(int i = 0; i < n; i++)
    for(int j = 0; j < n; j++)
      result(i,j) = (i == j? 1: 0);
  return result;
}




integer_matrix trans(const integer_matrix& M)
{ int d1 = M.dim1();
  int d2 = M.dim2();
  integer_matrix result(d2,d1);
  for(int i = 0; i < d2; i++)
    for(int j = 0; j < d1; j++)
      result.elem(i,j) = M(j,i);
  return result;
}



@ We now turn to the functions of linear algebra. We frequently need a function 
that swaps the values of its arguments.

@f T int

@<linear algebra@>=



template<class T>
void swap(T& x,T&y) 
{@+ T h = x; @+ x = y; @+ y = h;
@+ }



@ The most important operation is the solution of a non-homogeneous 
linear system. We use Gaussian elimination as described by 
J. Edmonds (J. Edmonds, Systems of distinct representatives and
linear algebra, Journal of Research of the Bureau of National 
Standards, (B), 71, 241 -245) .

Consider a linear system $A \cdot x = b$. Gaussian elimination operates in two
phases. In the first phase it transforms $A$ by a sequence of row 
and column operations
into an upper diagonal matrix and in the second phase it solves the 
resulting upper diagonal system. The first phase of Gaussian elimination
operates in subphases, numbered $0$ to $m-1$. 





@<linear algebra@>+=

bool linear_solver(const integer_matrix & A, const integer_vector& b, @\
integer_vector &x, number & D, integer_matrix & spanning_vectors, integer_vector & c) 
{

@<initialize |C| and |L| from |A| and |b|@>@;

  @<LU-decomposition@>@;

 
  @<test whether a solution exists and compute $c$ is there is no solution@>@;

  if (solvable) {@<compute solution space@>@;
                }
 


 
  return solvable;
}

@ Initialization.

@<initialize |C| and |L| ...@>=

int i,j,k; // indices to step through the matrix
 
  
    int rows = A.dim1();
    int cols = A.dim2();
   

if (b.dim()!= rows) error_handler(1,"linear_solver: b has wrong dimension");


integer_matrix C(rows,cols+1);// the matrix in which we will calculate ($C = (A\vert b)$)
  

  /* copy |A| and |b| into |C| */
  for(i=0;i<rows;i++) {
    for(j=0;j<cols;j++)
      {C(i,j)=A(i,j);
      }
    C(i,cols)=b[i];
  }

integer_matrix L(rows,rows);

for (i =0; i<rows; i++)
for(j=0;j<rows;j++) @+ L(i,j) = (i == j? 1 : 0);



@ LU-decomposition.

@<LU-decomposition@>=


 array<int> var(0,cols-1); // column $j$ of |C| represents the |var[j]|-th variable

 for(j=0;j<cols;j++)
    var[j]= j; // at the beginning, variable $x_j$ stands in column $j$



 number denom = 1;   // the determinant of an empty matrix is 1

 int sign = 1; // no interchanges yet

 int rank = 0; // we have not seen any non-zero row yet

 /* here comes the main loop */
 for(k=0;k<rows;k++){   
 
   @<search for a non-zero element; if found it is in $(i,j)$@>@;

   if (non_zero_found){

   rank++;   //increase the rank

   @<interchange rows $k$ and $i$ and columns $k$ and $j$@>@;

   @<one phase of LU-decomposition@>@;
  
   }

   else { @+ break; @+ }

 }



@ We search for a non-zero element. 

@<search for...@>=

   bool non_zero_found = false;

   for(i =k;i<rows;i++) { // step through rows $k$ to |rows-1|
    
     for (j = k ; j < cols && C(i,j) == 0; j++) ;
       // step through columns |k| to |cols - 1|
      
    if (j < cols) { @+ non_zero_found = true; @+ break;  @+}

   
   }

 

@ We interchange rows and columns. Any exchange changes the sign
of the determinant.

@<interchange rows ...@>=


if (i != k) {

sign = -sign;

  

  /* we interchange rows |k| and |i| of |L| and |C| */

swap(L[i],L[k]);
swap(C[i],C[k]);

}



if ( j != k){

sign = -sign;

   /* We interchange columns |k| and |j| */

   for( int ih =0;ih <rows;ih ++) swap(C(ih,k),C(ih,j));
   
	


   /* We store the interchanging of the variables in |var| */
  
  swap(var[k],var[j]);
}
 
@ Now we are ready to do the pivot-step with the element $C_{k,k}$. We do 
the $L$'s first since we want to work with
the old values of $C$.

@<one phase of LU-decomposition@>=
  
for(i = k+1; i < rows;i++)  {
   for (j = 0; j <  rows; j++)  //and all columns of |L|
       {L(i,j)  = (L(i,j) *C(k,k) - C(i,k)*L(k,j))/denom;
        
       }
    }
  

 for(i = k+1; i < rows;i++) 
 {      

/* the following iteration uses and changes |C(i,k)| */
        number temp = C(i,k);

   	for (j = k; j <= cols; j++)
       { 
   	C(i,j) = (C(i,j)*C(k,k) - temp*C(k,j))/denom;

       }
       
    
  }


denom = C(k,k);



@<check invariant after phase |k| of LU-decomposition@>@;

@ It is good custom to check the state of a computation. The matrix |L|
multiplied by |A| permuted as given by |var| should be
equal to the current |C|. The permutation |var| moves column |var[j]| of |A|
to column |j| of |C|. 

@<check invariant after phase |k| of LU-decomposition@>=

#ifdef TESTALL

for(i = 0; i <rows; i++)
{
  for (j = 0; j <cols; j++)
   { 
    number Sum = 0;
    for (int l = 0; l < rows; l++) Sum += L(i,l)*A(l, var[j]);
   
    if (Sum  != C(i,j)) 
             error_handler(1,"L*A*V different from C");


}
}
#endif




@  We are done with Gaussian elimination. At this point |C| has a 
$|rank| \times |rank|$ upper triangular matrix in its left upper corner and the remaining
rows of |C| are zero. The system is solvable if the current right hand side
has no non-zero entry in row |rank| and below. Assume otherwise, 
say |C(i,cols) != 0|. Then the $i$-th row of $B$ proves the unsolvability of the
system.

@<test whethe...@>=

  
bool solvable = true;
 
for(i =rank ;i <rows && C(i,cols) == 0;i++);  // no body

if (i < rows) 
	{solvable = false;@/
	c = integer_vector(rows);
        
        for (j = 0; j <rows; j++) c[j] = L(i,j);
        }

 

@ We compute the solution space. It is determined by a solution |x| of the
non-homogenoeus system plus  $|cols| - |rank|$ linearly independent
solutions to the homogeneous system. 

 Recall that |C| has a 
$|rank| \times |rank|$ upper triangular matrix
in its upper left corner. 
We view the variables corresponding to the first |rank| columns  as
dependent and the others as independent. The vector |var| tells us
the input variables corresponding to these columns. 


The components of |x| are 
rational numbers with the common denominator |denom|. 
The components corresponding to the independent variables are set to zero 
and the components corresponding to the dependent variables are 
computed by back substitution. In the back substitution code we 
compensate the fact that all variables have denominator |D| by 
premultiplying the righthand side with |D|.

We have
a spanning vector for each independent variable. We set the value 
of the dependent variable to $1 = D/D$
and then compute the value of all dependent variables.


@<compute solution space@>=


x = integer_vector(cols);
D = denom;

for(i=rank -1 ;i>= 0;i--) 
{ number h =C(i,cols) * D;  
  for (j = i+1; j<rank; j++) @+ h -= C(i,j)*x[var[j]];
  x[var[i]]= h / C(i,i);
}

#ifdef TEST
/* we check whether |x| is a solution */
{ for (i = 0; i < rows; i++)
  { number sum = 0;
    for (j =0; j < cols; j++) sum += A(i,j)*x[j];
    if (sum != D * b[i])
	error_handler(1,"linalg: base is not a solution");
  }
}
#endif

	
int dimension = cols-rank; //dimension of solution

spanning_vectors = integer_matrix(cols,dimension);
 
if (dimension > 0){

  

 

 
/* In the $l$-th spanning vector, $0 \le l < |dimension|$  we set  
variable |var[rank + l]| to $1 = |denom|/|denom|$ and then the 
dependent variables as dictated by the $|rank| + l$-th column of |C|.*/

  
for(int l=0;l < dimension;l++) {
 
    spanning_vectors(var[rank+l],l)=D;  

    for(i = rank - 1; i >= 0 ;i--)
    {
       
      number h = -C(i,rank + l)* D;  
  for ( j= i+1; j<rank; j++) @+ h -= C(i,j)*spanning_vectors(var[j],l);

     spanning_vectors(var[i],l)=h / C(i,i);


     }



#ifdef TEST  

/* we check whether the $l$-th spanning vector is a solution 
of the homogeneous system */
{ integer_vector zero(rows);
  if (A *spanning_vectors.col(l) != zero)
 	error_handler(1,"linalg: spanning_vector is not a solution");
}

#endif
}

}


@ This completes the implementation of the linear solver. We next turn
to the functions |determinant|, and |inverse|, and |rank|.

The determinant function is simple. The determinant is either zero (if the
matrix does not have full rank) or is |sign*denom| where |sign| and |denom|
are computed during LU-decomposition.


@<linear algebra@>+=


number determinant( const integer_matrix& A)
{
if (A.dim1() != A.dim2())
error_handler(1,"determinant: only square matrices are legal inputs");
integer_vector b(A.dim1());   // zero-vector
@<initialize |C| and  ...@>@;
@<LU-decomposition@>@;

if (rank < rows) return 0;
else
return sign * denom;

}

@ The certified version of the determinant is slightly more interesting. 
If the matrix is singular then the last row of |L| proves that fact, i.e.,
is a vector |c| with $c^T \cdot A = 0$. If the matrix is non-singular
then we return the LU-decomposition. Note that no row interchanges
took place during the first phase of Gaussian elimination when |A| is
non-singular. Thus |L| is really a lower diagonal matrix. 

We need to check whether |L| and |U| are really lower and upper 
diagonal
matrices, whether the diagonal elements are as prescribed, 
whether $L \cdot A \cdot Q = U$ and whether the claimed value of 
the determinant is equal to
the sign of $Q$ times $U(n-1,n-1)$. Since $Q$ is given implicitely by 
an |arrray<int> q| the handling of |Q| requires a little bit of work.

We first check whether |q| is indeed a permutation by checking 
whether it is surjective and then compute the sign of |q|. To 
do so we trace the cycles of
|q|. A cycle of length $l$ contributes $(-1)^{l-1}$ to the 
sign. We trace 
each cycle starting at its smallest element. 

@<linear algebra@>+=



number determinant( const integer_matrix& A, @\
integer_matrix & LD, integer_matrix & UD, array<int> & q, integer_vector & c)
{
if (A.dim1() != A.dim2())
error_handler(1,"determinant: only square matrices are legal inputs");
integer_vector b(A.dim1());   // zero-vector
@<initialize |C| and  ...@>@;
@<LU-decomposition@>@;

if (rank < rows)
{ c = L.row(rows-1);
  return 0;
}
else
{ LD = L; 
  UD = integer_matrix(rows,rows);
  for (i = 0; i < rows; i++) 
    for(j = 0; j <rows; j++) UD(i,j) = C(i,j);
  q = var;
return sign * denom;
}

}

bool verify_determinant(const integer_matrix& A, @\
number D,
integer_matrix & L, integer_matrix & U, array<int> q,
integer_vector & c)
{int n = A.dim1(); 
int i,j;
if (D == 0)
{ /* we have $c^T \cdot A = 0$  */
integer_vector zero(n);
return  (trans(A) * c == zero);
}
else
{ /* we check the conditions on |L| and |U| */
 if (L(0,0) != 1) return false;
 for (i = 0; i<n; i++)
 { for (j = 0; j < i; j++) if (U(i,j) != 0) return false;

    if (i > 0 && L(i,i) != U(i-1,i-1)) return false;

   for (j = i+1; j < n; j++) if (L(i,j) != 0) return false;
 }
/* check whether $L \cdot A \cdot Q = U$ */
integer_matrix LA= L * A;
for (j = 0; j < n; j++)
  if (LA.col(q[j]) != U.col(j)) return false;
/* compute sign |s| of |Q| */
int sign = 1;
/* we chase the cycles of |q|. An even length cycle contributes -1 and vice versa */

array<bool> already_considered(0,n-1);


for (i = 0; i < n; i++) @+ already_considered[i] = false;

for (i = 0; i < n; i++) @+ already_considered[q[i]] = true;

for (i = 0; i < n; i++) 
  if (! already_considered[i])
     error_handler(1,"q is not a permutation");
  else 
     already_considered[i] = false;


for (i = 0; i < n; i++)
{ if (already_considered[i]) continue;

  /* we have found a new cycle with minimal element $i$. */

int k = q[i];
already_considered[i] =true;

while (k != i)
{ sign = - sign;
  already_considered[k]= true;
  k = q[k];
}
  
}


return (D == sign * U(n-1,n-1));




}
}

@ |rank| and |inverse| are easy to implement. The rank is already computed 
during LU-decomposition and the $i$-th column of the inverse is a solution
to the linear system $A \cdot x = e_i$ where $e_i$ is the $i$-th unit-vector.


@<linear algebra@>+=

int rank(const integer_matrix& A)
{
if (A.dim1() != A.dim2())
error_handler(1,"rank: only square matrices are legal inputs");
integer_vector b(A.dim1());   // zero-vector
@<initialize |C| and ...@>@;
@<LU-decomposition@>@;
return rank;

}




bool inverse(const integer_matrix& A, @/
integer_matrix& inverse, number& D,@/
integer_vector& c)
{
if (A.dim1() != A.dim2())
error_handler(1,"inverse: only square matrices are legal inputs");
integer_vector b(A.dim1());   // zero-vector
@<initialize |C| and ...@>@;
@<LU-decomposition@>@;

if (rank < rows)
{ /* matrix is singular; we return a vector $c$ with $c^T \cdot A = 0$.  */


c = integer_vector(rows);
        
for (j = 0; j <rows; j++) c[j] = L(rows - 1,j);


return false;
}

@<complete computation of inverse@>;

return true;



}



@ In order to  compute the $i$-th column of the inverse we need to solve the
linear system $A\cdot x = e_i$, where $e_i$ is the $i$-th unit vector. We
have already subjected $e_i$ to the first phase of Gaussian elimination
as the $i$-th column of $L$. Thus we only have to perform back substitution with
this column in order to get the $i$-th column of the inverse. 

@<complete computation of inverse@>=


inverse = integer_matrix(rows,rows);

for(i = 0; i <rows; i++) {  // $i$-th column of inverse
    
   for (j = rows -1; j >= 0; j--) 

	{ number h = L (j,i) * D;  
 	 for (int l = j+1; l<rows; l++) @+ h -= C(j,l)*inverse(var[l],i);
  	inverse(var[j],i) = h / C(j,j);
	}

}  

#ifdef TEST
if (!(A * inverse == identity(rows) * D))
error_handler(1,"matrix inverse computed incorrectly");
#endif      




@ Let's next solve a homogeneous system. 

@<linear algebra@>+=



bool homogeneous_linear_solver(const integer_matrix &A,@/
integer_vector& x)
/* returns true if the homogeneous system $Ax = 0$ has a non-trivial solution
and false otherwise. */
{
integer_vector b(A.dim1());   // zero-vector
number D;
@<initialize |C| and ...@>@;
@<LU-decomposition@>@;

integer_matrix spanning_vectors;


@<compute solution space@>;
if (dimension == 0) return false;
/* return first column of |spanning_vectors| */
for (i = 0; i < cols; i++) @+ x[i] = spanning_vectors(i,0);
return true;
}


@
\end{document}
