@* The Basic Structure of the Program.
We want to separate our program in three parts: a header file
containing all definitions for inclusion in other programs, a
code file containing all implementations, and a short demo
program mainly used for debugging and presentation.
The fundamental data structures for the simplicial complex will be
called |class Simplex| and |class Triangulation|. With these terms, we
can give a short overview of the program now.

@ These parts go into the header file.
@(chull.h@>=
#ifndef CHULL_H
#define CHULL_H
@<Header files to be included@>@;
@<class Triangulation@>@;
#endif

@ This is everything that produces code.
@c
@<class Simplex@>@;
@<Member functions of class Triangulation@>@;
@<Friend functions of class Triangulation@>@;

@ Our demo routine.
@(main.c@>=
@<Main program@>


@
From {\em LEDA}  we use the data types |array|, |list|, |h_array|,
|integer|, |vector|, |dictionary| and we use
streams for I/O.

@<Header files to be included@>=
#include <LEDA/array.h>
#include <LEDA/list.h>
#include <LEDA/h_array.h>
#include <LEDA/integer.h>
#include <LEDA/vector.h>
#include <LEDA/dictionary.h>
#include <LEDA/stream.h>


@ In order to show triangulations on the screen, we
 implement a function that draws  the triangulation onto the screen in the
two dimensional case using {\em LEDA}'s
|window| type. Therefore, we have to include the appropriate {\em LEDA}
header files. We are working with the X11R5 (xview) window environment.
@<Header files to be included@>+=
#include <LEDA/window.h>
#include <LEDA/plane.h>

@ We use the datatypes |d_rat_point| and |hyperplane|\footnote{Many
thanks to Kurt Mehlhorn for giving me working prelimary versions of them.}
for numerically correct computations in conjunction with their basic
type ({\sc Leda}-)|integer| where
we need to. They provide easy to use member- and friend-functions hiding several
Gaussian eliminations from the code. They will be integrated into {\sc
Leda} soon. 

@<Header files to be...@>+=
#include "d_rat_point.h"
#include "hyperplane.h"

@* The Fundamental Data Structures.

Now we can begin to define our fundamental data structures (cf.\
Section \ref{how to store things}).
The whole simplicial complex will be managed by the |class
Triangulation|.  In this class, we store the coordinate vectors of the
points given so far (|list| |coordinates|), the dimension of the
convex hull of these points (|int dcur|), the dimension of the
coordinate vectors of the input points (|int dmax|) and a pointer to
the origin simplex, from which we can reach all other simplices.
When we compute the equation for the base facet
of an unbounded simplex, it is useful to know a point which lies in the
interior of the origin simplex (cf.\ Section \ref{compute hyperplanes})
and we also need such a point as a starting point for the segment
walking method.  An appropriate point is the center point of the
origin simplex
\[O= \sum_{i=0}^{dcur} \frac{v_i}{dcur+1},\]
where
$v_0,\cdots,v_{dcur}$ are the coordinate vectors of the vertices of
the origin simplex.  To avoid the numerically problematic division, we
store in the variable |quasi_center| only the sum of the $v_i$'s and when we need $O$, we
have to remind \label{center_comp}that $O = $|quasi_center|$ / %\linebreak
(dcur + 1)$.
Furthermore, we store a |list| of all constructed simplices
(|all_simplices|) which makes it easier to traverse all simplices (for
instance in the destructor of the class or when displaying the
simplicial complex).  With this list, the interested reader may
implement a more efficient copy constructor for the class.

During the insertion of some $x_i$, we have to find the $x_i$-visible
facets of $CH(\pi_{i-1})$.  For this purpose, we have implemented
three search methods: the visibility search method and the segment
walking method described in \cite{CMS} and a modification of the
visibility search method.  For the selection of the search method, we
introduce an enumeration type with the elements
|VISIBILITY|, \allowbreak |MODIFIED_VISIBILITY| and |SEGMENT_WALK|, respectively.

The public member |method| of |Triangulation| determines the search
\linebreak
method to be used; it can be changed by the user at any time and
its default value is |SEGMENT_WALK|.
Each of these search methods stores its result (i.e., pointers to
the unbounded simplices having the $x_i$-visible facets of
$CH(\pi_{i-1})$ as base facets) in the list
|visible_simplices|.

As the  representation of $\aff R$, we use the (affine linearly
independent) vertices of the origin simplex.

\begin{zitat}{\cite{CMS}, p. 12}
In order to support the efficient computation of the set $R(x)$\footnote{
This set is explained in the discussion of the deletion process (section
\ref{del_proc})} we need to
augment our data structure slightly. We assume that each point stores a
pointer to some simplex containing it and that every simplex stores a list
of the points contained in it.
\end{zitat}
The pointers to the simplex for each point are stored in a {\sc Leda} |h_array simplex|.
We use |h_array|s for the insertion order of the points in the list
|coordinates|, and to store the position of |x| in the |points|-list iff |x|
is an interior point (|nil| if it is a vertex).
These |h_array|s are mainly of use when deleting a point.


@<class Triangulation@>=

class Simplex; // forward reference

enum search_method {VISIBILITY, SEGMENT_WALK, MODIFIED_VISIBILITY};
@#

class Triangulation {
	private:@/

	list<d_rat_point> coordinates;
	       // the coordinate vectors of the $x_i$
	dictionary<d_rat_point, list<list_item> *> co_index;
	       // the index in coordinates for each point
	int co_nr; // current max. order nr for the point in |coordinates|
	h_array<list_item,int> order_nr; //order number for each point
	h_array<list_item,Simplex *> simplex; // a simplex |x| belongs to
	h_array<list_item,list_item> position;
	 // position of |x| in |simplex[item_x]->points|, |nil| if |x| is a vertex

	int dcur; // dimension of the current convex hull
	int dmax; // dimension of the coordinate vectors
	Simplex *origin_simplex; // pointer to the origin simplex
	d_rat_point quasi_center;
	// sum of the coordinate vectors of the vertices of the origin simplex
	list<Simplex*> all_simplices; // list of all simplices

	@<Further member declarations of |class Triangulation|@>@;

        void print(Simplex *); // writes some statistics about |S| to |stdout|

	public:@/

        search_method method;
	int searched_simplices; // used for statistical reasons
	int created_simplices();
	     // returns the number of simplices that have been created

	void insert(const d_rat_point &x); // insertion routine
	bool member(const d_rat_point &x); // to test if
	void del(const d_rat_point &x); // deletion routine
	d_rat_point find_closest_point(const d_rat_point &x); // for mouse-convenience
        void show(window &W); // draws the triangulation onto the screen
	void print_all();   // calls print() for all simplices
	void print_extremes(ostream &o); // print outer points
	list<d_rat_point> points(); // return coordinates
	Triangulation(int d=0, search_method m=SEGMENT_WALK);
	   // constructor function with default arguments
	~Triangulation(); // destructor function
	Triangulation(Triangulation &T); // copy-constructor

	Triangulation &operator=(const Triangulation &T);
	//currently disabled
friend ostream &operator<<(ostream&,Triangulation&); // I/O for
friend istream &operator>>(istream&,Triangulation&); // Triangulations
	};

@
At the end of the program we want to be able to print the number of
simplices that have been created.
If a simplex is the $k$-th simplex created, its component |sim_nr|
gets $k$ (cf.\ |Simplex::Simplex()|).
@<Member fun...@>=
int Triangulation::created_simplices()
{ Simplex Dummy(2);
  static dummys_created=0;

  dummys_created++;
  return Dummy.sim_nr - dummys_created;
}

@
The constructors for |class Triangulation| are easy to implement. The
default search method is segment walking.
@<Member fun...@>+=
Triangulation::Triangulation(int d, search_method m) :
 order_nr(-1), simplex(nil), position(nil)
{
   co_nr = 0;
   dcur = -1;
   dmax = d;
   searched_simplices = 0;
   origin_simplex = nil;
   method=m;
   inner_simplex = nil;
}

Triangulation::Triangulation(Triangulation &T) :
 order_nr(-1), simplex(nil), position(nil)
{
d_rat_point v;

  co_nr = 0;
  dcur=-1;
  dmax=T.dmax;
  searched_simplices = 0;
  origin_simplex = nil;
  method=T.method;
  inner_simplex=nil;
  forall(v,T.coordinates)
    insert(v);		// this is the actual copying
} 

@
In the destructor for |Triangulation|, we have to release the storage
which was allocated for the simplices.
@<Member fun...@>+=
Triangulation::~Triangulation()
{ Simplex *S;
  forall(S, all_simplices)
    delete(S);
}
	
@
Now we define the |class Simplex|.
We make |class Triangulation| a friend of |class Simplex|, so that it
can access every private member of |class Simplex|.
\label{class Simplex sec}For each simplex, we store its vertices as an array
|vertices| of pointers to the corresponding occurrences in the
list |coordinates| of |class Triangulation|.
For the anti-origin we store |nil|.
The array has length |dmax+1| since a simplex has at most
|dmax+1| vertices.
When the current hull has dimension |dcur|, only the array elements 0 to
|dcur| are used.
Furthermore, we use the following convention:

\smallskip
\centerline{the peak vertex of the simplex is
always |vertices[0]|.}
\smallbreak
In order to represent the neighborhood relation, we use a second array 
|neighbors|, such that |neighbors[k]| points to
the simplex opposite to the vertex |vertices[k]|.

Given a vertex $v$ of a simplex $V$, let $W$ be the neighbor
of $V$ opposite to $v$.
It is also useful to find the vertex $w$ opposite to $v$,
i.e., the vertex $w$ of $W$ which is not a vertex of $V$.
For this purpose, we use an array |opposite_vertices|:
 if $v$ is the $k$-th vertex of $V$, i.e., |V->vertices[k]==v|, and
$w$ is the $l$-th vertex of $W$, then |V->opposite_vertices[k]==l|
and vice versa |W->opposite_vertices[l]==k|.
\begin{figure}[hbtp]
\begin{center}
\input{simplex-connect.latex}
\end{center}
\caption{\label{simplex-connect}The connection of two simplices $V$
and $W$}
\end{figure}

Figure \ref{simplex-connect} illustrates the connection between two
adjacent simplices $V$ and $W$.  The numbers that stand outside the
simplices are the numbers of the vertices of $V$, the others being the
numbers of the vertices of $W$. In both simplices, the vertex with
number $0$ is the peak vertex.  The connectivity of $V$ and $W$ is
expressed as follows: we have |V->neighbors[2]==W| and
|W->neighbors[0]==V|, indicated by the corresponding arrows.
Furthermore, we have |V->opposite_vertices[2]==0| and vice versa
|W->opposite_vertices[0]==2|.

For the test whether a point sees a facet of a given simplex, we need
the hyperplane which contains the facet.  The normal
vector must be oriented in such a way that the vertex opposite to the face lies
in the positive halfspace.  When we need the equation of a hyperplane for a
facet $i$ of a simplex |S| (i.e, the facet opposite to the $i$-th
vertex of $S$), we call the function |compute_plane()| (or more often
|sees_facet()|, which is more comfortable and often suffices.)
 (cf.\ Section~\ref{compute hyperplanes}),
which are members of |class Triangulation|.
Once we have computed these values
for a facet, we store them in the array |facets| 
 of the corresponding simplex so that they not have to
be computed again when they are used the next time. Unfortunately,
after a dimension jump all entries of |facets| 
become invalid.  Therefore, we store in an array
|valid_equations| the time, i.e.\ the current dimension |dcur| of the
convex hull, when the value of the facet's equation was computed.
The functions |sees_facet()| and |compute_plane()| check whether the
respective values of the $i$-th facet of simplex |S| are still valid
and if not they compute them.  Then they return the valid values.  The
values are invalid iff |valid_equations[i]!=dcur|. Then the values are
computed new and |valid_equations[i]| is set to |dcur|.  Initially,
they are set to $-1$ to denote that none are valid.

The |list points| holds the position of the inner points of the simplex
(points inserted after the peak vertex that lie in the simplex).

In the implementation of the deletion process we must not
forget that we may have to set |valid_equations[i]| to $-1$ again for
all $i$ and all simplices if we delete a vertex which was a dimension
jump. This sounds expensive, but we have to look anyway at all vertices of
all simplices of the remaining hull when we reduce the dimension,
so invalidating the plane
equations adds only a constant amount of time.

We also need a mark to indicate visited simplices when we traverse
the triangulation (e.g., for the visibility search or for traversing
all simplices when we process a dimension jump).

In the deletion process we find it useful to have the |list_item|
of a simplex handy for updating the |list all_simplices|.

@d anti_origin nil

@<class Simplex@>=

#include "chull.h"

class Simplex {
	friend class Triangulation; // |Triangulation| has unrestricted access
	private:@/
	int sim_nr; // useful for debugging; unique number for each simplex
	list_item this_item; // points into |all_simplices|, for easy deletion
	array<list_item> vertices; 
           // pointers to the coordinate vectors of the vertices
	array<Simplex *> neighbors; 
	array<int> opposite_vertices; // indices of opposite vertices

	array<hyperplane> facets;
	array<int> valid_equations;
	// dimension in which corresponding hyperplane was computed

	bool visited;
	// used to mark simplices when traversing the triangulation
	list<list_item> points; // pointer into |coordinates| of other points located in this vertex

	Simplex(int dmax); // constructor function

	~Simplex()@+ {}@+; // destructor function

LEDA_MEMORY(Simplex);
	};

@
The constructor for |class Simplex| sets the size of the
arrays and marks the simplex
as not visited.
@<class Simplex@>+=

Simplex::Simplex(int dmax) : vertices(0,dmax),
		     neighbors(0,dmax),
		     opposite_vertices(0,dmax),
		     facets(0,dmax),
		     valid_equations(0,dmax)
{ static int lfdnr = 0; // each simplex gets a unique number (for debugging)

  sim_nr = lfdnr++;
  for (int i = 0; i <= dmax; i++)  {
    neighbors[i] = nil; // to avoid illegal pointers when using |print()|
    valid_equations[i] = -1;
  }
  visited = false;
}

